#!/usr/bin/env python3
"""
SkhiBridgX Desktop Analytics - Ultra-Modern Cool GUI Framework
Professional Tkinter application with modern design, animations, and cool effects
"""

import tkinter as tk
from tkinter import ttk, filedialog, messagebox, scrolledtext
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import seaborn as sns
from datetime import datetime, timedelta
import os
import json
import threading
from abc import ABC, abstractmethod
import random

class ModernButton(tk.Button):
    """Custom button with hover effects and modern styling"""
    def __init__(self, parent, **kwargs):
        self.default_bg = kwargs.get('bg', '#3b82f6')
        self.hover_bg = self._lighten_color(self.default_bg, 20)
        self.click_bg = self._darken_color(self.default_bg, 20)
        
        # Set default modern styling
        modern_style = {
            'relief': 'flat',
            'bd': 0,
            'padx': 20,
            'pady': 12,
            'font': ('Segoe UI', 11, 'bold'),
            'cursor': 'hand2'
        }
        modern_style.update(kwargs)
        
        super().__init__(parent, **modern_style)
        
        # Bind hover events
        self.bind("<Enter>", self.on_enter)
        self.bind("<Leave>", self.on_leave)
        self.bind("<ButtonPress-1>", self.on_click)
        self.bind("<ButtonRelease-1>", self.on_release)
    
    def _lighten_color(self, color, percent):
        """Lighten a hex color by percentage"""
        if not color.startswith('#'):
            return color
        try:
            rgb = tuple(int(color[i:i+2], 16) for i in (1, 3, 5))
            rgb = tuple(min(255, int(c + (255 - c) * percent / 100)) for c in rgb)
            return f"#{rgb[0]:02x}{rgb[1]:02x}{rgb[2]:02x}"
        except:
            return color
    
    def _darken_color(self, color, percent):
        """Darken a hex color by percentage"""
        if not color.startswith('#'):
            return color
        try:
            rgb = tuple(int(color[i:i+2], 16) for i in (1, 3, 5))
            rgb = tuple(max(0, int(c * (100 - percent) / 100)) for c in rgb)
            return f"#{rgb[0]:02x}{rgb[1]:02x}{rgb[2]:02x}"
        except:
            return color
    
    def on_enter(self, e):
        self.config(bg=self.hover_bg)
    
    def on_leave(self, e):
        self.config(bg=self.default_bg)
    
    def on_click(self, e):
        self.config(bg=self.click_bg)
    
    def on_release(self, e):
        self.config(bg=self.hover_bg)

class AnimatedLabel(tk.Label):
    """Label with typewriter animation effect"""
    def __init__(self, parent, text="", speed=50, **kwargs):
        self.full_text = text
        self.speed = speed
        self.current_text = ""
        self.index = 0
        
        super().__init__(parent, text="", **kwargs)
        
    def animate_text(self):
        """Animate text with typewriter effect"""
        if self.index < len(self.full_text):
            self.current_text += self.full_text[self.index]
            self.config(text=self.current_text)
            self.index += 1
            self.after(self.speed, self.animate_text)

class CoolSectorApp(ABC):
    """Ultra-modern base class for sector-specific analytics applications"""
    
    def __init__(self, root):
        self.root = root
        self.df = None
        self.reports_generated = []
        self.sector_name = self.get_sector_name()
        self.sector_color = self.get_sector_color()
        self.accent_color = self.get_accent_color()
        
        # Configure main window with modern styling
        self.root.title(f"🚀 SkhiBridgX {self.sector_name} Analytics Pro - PREMIUM")
        self.root.configure(bg='#0f0f23')  # Deep space blue
        
        # Set modern window properties
        self.root.attributes('-alpha', 0.98)  # Slight transparency
        
        # Responsive window sizing
        screen_width = self.root.winfo_screenwidth()
        screen_height = self.root.winfo_screenheight()
        
        window_width = min(int(screen_width * 0.85), 1600)
        window_height = min(int(screen_height * 0.85), 1000)
        
        x = (screen_width - window_width) // 2
        y = (screen_height - window_height) // 2
        
        self.root.geometry(f"{window_width}x{window_height}+{x}+{y}")
        self.root.minsize(1200, 700)
        
        # Initialize animations and effects
        self.setup_modern_ui()
        self.populate_report_list()
        self.start_welcome_animation()
    
    @abstractmethod
    def get_sector_name(self) -> str:
        pass
    
    @abstractmethod
    def get_sector_color(self) -> str:
        pass
    
    def get_accent_color(self) -> str:
        """Return accent color for highlights"""
        accent_map = {
            'Insurance': '#fbbf24',
            'Real Estate': '#10b981',
            'CPA': '#8b5cf6'
        }
        return accent_map.get(self.sector_name, '#06b6d4')
    
    @abstractmethod
    def get_report_list(self) -> list:
        pass
    
    @abstractmethod
    def load_sample_data(self):
        pass
    
    @abstractmethod
    def perform_sector_analysis(self):
        pass

    def setup_modern_ui(self):
        """Create ultra-modern UI with animations and effects"""
        # Main container with gradient-like effect
        main_container = tk.Frame(self.root, bg='#0f0f23')
        main_container.pack(fill='both', expand=True)
        
        # Animated header section
        self.setup_animated_header(main_container)
        
        # Content area with glass-morphism effect
        content_frame = tk.Frame(main_container, bg='#1a1a2e')
        content_frame.pack(fill='both', expand=True, padx=20, pady=10)
        
        # Left panel - Modern control center
        left_panel = tk.Frame(content_frame, bg='#16213e', relief='flat', bd=0)
        left_panel.pack(side='left', fill='both', padx=(0, 15), pady=15)
        left_panel.configure(width=450)
        
        # Right panel - Results dashboard
        right_panel = tk.Frame(content_frame, bg='#16213e', relief='flat', bd=0)
        right_panel.pack(side='right', fill='both', expand=True, pady=15)
        
        self.setup_modern_left_panel(left_panel)
        self.setup_modern_right_panel(right_panel)
        self.setup_modern_status_bar()

    def setup_animated_header(self, parent):
        """Setup animated header with cool effects"""
        header_frame = tk.Frame(parent, bg='#0f0f23', height=120)
        header_frame.pack(fill='x', pady=(20, 10))
        header_frame.pack_propagate(False)
        
        # Main title with glow effect
        title_text = f"{self.get_sector_icon()} SkhiBridgX {self.sector_name} Analytics"
        self.main_title = tk.Label(header_frame,
                                  text=title_text,
                                  font=('Segoe UI', 28, 'bold'),
                                  fg=self.sector_color,
                                  bg='#0f0f23')
        self.main_title.pack(pady=(10, 5))
        
        # Animated subtitle
        self.subtitle = AnimatedLabel(header_frame,
                                     text=f"⚡ Premium {self.sector_name} Analytics • 25+ AI-Powered Reports • Real-Time Processing",
                                     font=('Segoe UI', 12),
                                     fg=self.accent_color,
                                     bg='#0f0f23',
                                     speed=30)
        self.subtitle.pack()
        
        # Version badge
        version_frame = tk.Frame(header_frame, bg='#0f0f23')
        version_frame.pack(pady=(5, 0))
        
        version_badge = tk.Label(version_frame,
                                text="🎯 PREMIUM VERSION",
                                font=('Segoe UI', 10, 'bold'),
                                fg='#ffffff',
                                bg='#ef4444',
                                padx=15, pady=5)
        version_badge.pack()

    def setup_modern_left_panel(self, parent):
        """Setup modern control panel with glass-morphism cards"""
        # Data Input Card
        data_card = self.create_glass_card(parent, "📊 Data Management", self.accent_color)
        
        # Modern file load button
        load_btn = ModernButton(data_card,
                               text=f"📂 Upload {self.sector_name} Data",
                               command=self.load_csv_file,
                               bg=self.sector_color,
                               fg='white',
                               width=30)
        load_btn.pack(pady=(10, 5), padx=15, fill='x')
        
        self.file_status = tk.Label(data_card,
                                   text="• No data loaded",
                                   font=('Segoe UI', 10),
                                   fg='#94a3b8',
                                   bg='#16213e')
        self.file_status.pack(pady=(0, 15), padx=15)
        
        # Sample data button
        sample_btn = ModernButton(data_card,
                                 text=f"🎯 Load Sample Data",
                                 command=self.load_sample_data,
                                 bg='#8b5cf6',
                                 fg='white')
        sample_btn.pack(pady=(0, 15), padx=15, fill='x')
        
        # AI Magic Button Card
        magic_card = self.create_glass_card(parent, "🤖 AI Magic Button", '#ef4444')
        
        magic_btn = ModernButton(magic_card,
                                text=f"🔮 {self.get_magic_button_text()}",
                                command=self.magic_button_action,
                                bg='#ef4444',
                                fg='white')
        magic_btn.pack(pady=(10, 15), padx=15, fill='x')
        
        # Reports Card
        reports_card = self.create_glass_card(parent, "📈 Analytics Suite", self.sector_color)
        
        # Modern report list
        list_frame = tk.Frame(reports_card, bg='#16213e')
        list_frame.pack(fill='both', expand=True, padx=15, pady=10)
        
        scrollbar = tk.Scrollbar(list_frame, bg='#374151', troughcolor='#1f2937')
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        
        self.report_listbox = tk.Listbox(list_frame,
                                        font=('Segoe UI', 10),
                                        bg='#1f2937',
                                        fg='white',
                                        selectbackground=self.sector_color,
                                        selectforeground='white',
                                        bd=0,
                                        highlightthickness=0,
                                        yscrollcommand=scrollbar.set,
                                        height=8)
        self.report_listbox.pack(side=tk.LEFT, fill='both', expand=True)
        scrollbar.config(command=self.report_listbox.yview)
        
        # Generate button with pulse effect
        self.generate_btn = ModernButton(reports_card,
                                        text="🚀 Generate All Reports (25+)",
                                        command=self.generate_all_reports_threaded,
                                        bg='#ef4444',
                                        fg='white')
        self.generate_btn.pack(pady=(10, 15), padx=15, fill='x')

    def create_glass_card(self, parent, title, accent_color):
        """Create a glass-morphism style card"""
        card_frame = tk.Frame(parent, bg='#16213e', relief='flat', bd=1)
        card_frame.pack(fill='x', padx=15, pady=10)
        
        # Card header
        header = tk.Frame(card_frame, bg='#1e293b', height=40)
        header.pack(fill='x')
        header.pack_propagate(False)
        
        title_label = tk.Label(header,
                              text=title,
                              font=('Segoe UI', 12, 'bold'),
                              fg=accent_color,
                              bg='#1e293b')
        title_label.pack(pady=10)
        
        return card_frame

    def setup_modern_right_panel(self, parent):
        """Setup modern results dashboard"""
        # Results card
        results_card = self.create_glass_card(parent, "📊 Live Analytics Dashboard", self.accent_color)
        
        # Modern text area with syntax highlighting effect
        text_frame = tk.Frame(results_card, bg='#16213e')
        text_frame.pack(fill='both', expand=True, padx=15, pady=10)
        
        self.results_text = scrolledtext.ScrolledText(text_frame,
                                                     font=('JetBrains Mono', 10),
                                                     bg='#0d1117',
                                                     fg='#58a6ff',
                                                     insertbackground='#58a6ff',
                                                     wrap=tk.WORD,
                                                     bd=0,
                                                     highlightthickness=1,
                                                     highlightcolor=self.accent_color,
                                                     selectbackground='#264f78')
        self.results_text.pack(fill='both', expand=True)
        
        # Modern export controls
        export_frame = tk.Frame(results_card, bg='#16213e')
        export_frame.pack(fill='x', padx=15, pady=(0, 15))
        
        # Export buttons with icons
        ModernButton(export_frame, text="⚡ Quick Export", 
                    bg='#8b5cf6', fg='white',
                    command=self.quick_export).pack(side='left', padx=5)
        
        ModernButton(export_frame, text="📄 PDF Report", 
                    bg='#06b6d4', fg='white',
                    command=lambda: self.export_results('pdf')).pack(side='left', padx=5)
        
        ModernButton(export_frame, text="📊 Excel Data", 
                    bg='#10b981', fg='white',
                    command=lambda: self.export_results('xlsx')).pack(side='left', padx=5)
        
        ModernButton(export_frame, text="🌐 Web Report", 
                    bg='#f59e0b', fg='white',
                    command=lambda: self.export_results('html')).pack(side='left', padx=5)

    def setup_modern_status_bar(self):
        """Setup modern status bar with live updates"""
        status_frame = tk.Frame(self.root, bg='#0f0f23', height=30)
        status_frame.pack(side='bottom', fill='x')
        status_frame.pack_propagate(False)
        
        self.status_label = tk.Label(status_frame,
                                   text=f"🟢 {self.sector_name} Analytics Ready",
                                   font=('Segoe UI', 10),
                                   fg=self.sector_color,
                                   bg='#0f0f23')
        self.status_label.pack(side='left', padx=15, pady=5)
        
        # Live time display
        self.time_label = tk.Label(status_frame,
                                  text="",
                                  font=('Segoe UI', 10),
                                  fg='#64748b',
                                  bg='#0f0f23')
        self.time_label.pack(side='right', padx=15, pady=5)
        self.update_time()

    def update_time(self):
        """Update time display every second"""
        current_time = datetime.now().strftime("%H:%M:%S")
        self.time_label.config(text=f"🕐 {current_time}")
        self.root.after(1000, self.update_time)

    def start_welcome_animation(self):
        """Start welcome animation sequence"""
        self.root.after(500, self.subtitle.animate_text)
        self.log_message("🚀 Welcome to SkhiBridgX Premium Analytics!")
        self.log_message(f"⚡ {self.sector_name} Analytics Engine Initialized")
        self.log_message("🎯 Ready for data processing and report generation")

    def get_sector_icon(self) -> str:
        """Return sector-specific emoji icon"""
        icons = {
            'Insurance': '🛡️',
            'Real Estate': '🏠', 
            'CPA': '💼'
        }
        return icons.get(self.sector_name, '⚡')

    def get_magic_button_text(self) -> str:
        """Return sector-specific magic button text"""
        magic_texts = {
            'Insurance': 'AI State Filing Assistant',
            'Real Estate': 'AI CMA Generator', 
            'CPA': 'AI Tax Optimizer'
        }
        return magic_texts.get(self.sector_name, 'AI Analytics')

    def animate_button_click(self, button):
        """Animate button click with pulse effect"""
        original_bg = button.cget('bg')
        button.config(bg='#ffffff')
        self.root.after(100, lambda: button.config(bg=original_bg))

    def log_message(self, message, level='info'):
        """Enhanced logging with color coding"""
        timestamp = datetime.now().strftime("%H:%M:%S")
        
        # Color coding based on message type
        colors = {
            'success': '#22c55e',
            'warning': '#f59e0b', 
            'error': '#ef4444',
            'info': '#58a6ff',
            'magic': '#a855f7'
        }
        
        # Detect message type from content
        if '✅' in message or 'Success' in message or 'Complete' in message:
            level = 'success'
        elif '⚠️' in message or 'Warning' in message:
            level = 'warning'
        elif '❌' in message or 'Error' in message or 'Failed' in message:
            level = 'error'
        elif '🔮' in message or 'Magic' in message or 'AI' in message:
            level = 'magic'
        
        color = colors.get(level, '#58a6ff')
        
        def update_ui():
            # Insert with color
            self.results_text.tag_configure(level, foreground=color)
            formatted_message = f"[{timestamp}] {message}\n"
            self.results_text.insert(tk.END, formatted_message, level)
            self.results_text.see(tk.END)
        
        self.root.after(0, update_ui)

    # Add all the other methods from the original base_app.py
    # (load_csv_file, generate_reports, export functions, etc.)
    
    def populate_report_list(self):
        """Populate the report list with sector-specific reports"""
        reports = self.get_report_list()
        for report in reports:
            self.report_listbox.insert(tk.END, report)

    def load_csv_file(self):
        """Enhanced file loading with progress animation"""
        filetypes = [
            ("Data files", "*.csv;*.xlsx;*.xls"),
            ("CSV files", "*.csv"),
            ("Excel files", "*.xlsx;*.xls"),
            ("All files", "*.*")
        ]
        
        file_path = filedialog.askopenfilename(
            title=f"Select {self.sector_name} Data File",
            filetypes=filetypes
        )
        
        if file_path:
            self.log_message(f"📁 Loading file: {os.path.basename(file_path)}")
            self.status_label.config(text="🔄 Processing file...")
            
            try:
                file_ext = os.path.splitext(file_path)[1].lower()
                
                if file_ext == '.csv':
                    encodings_to_try = ['utf-8', 'latin-1', 'cp1252', 'iso-8859-1']
                    df_loaded = False
                    
                    for encoding in encodings_to_try:
                        try:
                            self.df = pd.read_csv(file_path, encoding=encoding)
                            self.log_message(f"✅ CSV loaded with {encoding} encoding", 'success')
                            df_loaded = True
                            break
                        except UnicodeDecodeError:
                            continue
                    
                    if not df_loaded:
                        raise Exception("Could not load CSV with any encoding")
                        
                elif file_ext in ['.xlsx', '.xls']:
                    self.df = pd.read_excel(file_path)
                    self.log_message("✅ Excel file loaded successfully", 'success')
                
                if self.df.empty:
                    raise Exception("File appears to be empty")
                
                self.df.columns = self.df.columns.str.strip()
                
                file_size = os.path.getsize(file_path) / 1024
                self.file_status.config(text=f"• {os.path.basename(file_path)} ({len(self.df)} rows)")
                self.status_label.config(text=f"✅ Data loaded: {len(self.df)} records")
                self.log_message(f"📊 Data shape: {self.df.shape[0]} rows × {self.df.shape[1]} columns", 'success')
                self.log_message(f"💾 File size: {file_size:.1f} KB")
                
                # Show data preview
                self.log_message("\n📋 DATA PREVIEW:")
                preview = self.df.head(3).to_string(max_cols=6, max_colwidth=15)
                for line in preview.split('\n'):
                    self.log_message(f"   {line}")
                
            except Exception as e:
                self.log_message(f"❌ Error loading file: {str(e)}", 'error')
                self.file_status.config(text="• File load failed")
                self.status_label.config(text="❌ File load failed")

    def quick_export(self):
        """Quick export with visual feedback"""
        if not self.reports_generated:
            messagebox.showwarning("No Results", "Please generate reports first")
            return
        
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        export_path = os.path.join(os.getcwd(), f"{self.sector_name}_Analytics_{timestamp}")
        
        self.log_message("⚡ Quick export starting...", 'magic')
        self.status_label.config(text="📤 Exporting reports...")
        
        try:
            os.makedirs(export_path, exist_ok=True)
            
            # Export all formats (placeholder - implement actual export)
            self.log_message(f"✅ Reports exported to: {export_path}", 'success')
            self.status_label.config(text="✅ Export complete!")
            
        except Exception as e:
            self.log_message(f"❌ Export failed: {str(e)}", 'error')

    def generate_all_reports_threaded(self):
        """Generate reports with enhanced visual feedback"""
        if self.df is None:
            messagebox.showwarning("No Data", "Please load data first")
            return
        
        self.generate_btn.config(text="🔄 Generating...", state='disabled')
        self.status_label.config(text="🚀 Generating 25+ reports...")
        
        thread = threading.Thread(target=self.generate_all_reports, daemon=True)
        thread.start()

    def generate_all_reports(self):
        """Generate reports with progress updates"""
        self.reports_generated = []
        
        try:
            self.log_message("🚀 Analytics engine starting...", 'magic')
            self.perform_sector_analysis()
            
            self.root.after(0, lambda: self.generate_btn.config(
                text="🚀 Generate All Reports (25+)", state='normal'))
            self.root.after(0, lambda: self.status_label.config(
                text=f"✅ Generated {len(self.reports_generated)} reports!"))
            
        except Exception as e:
            self.log_message(f"❌ Analysis failed: {str(e)}", 'error')
            self.root.after(0, lambda: self.generate_btn.config(
                text="🚀 Generate All Reports (25+)", state='normal'))

    def magic_button_action(self):
        """Enhanced magic button with animations"""
        if self.df is None:
            messagebox.showwarning("No Data", "Please load data first")
            return
        
        self.log_message(f"🔮 AI Magic Button Activated: {self.get_magic_button_text()}", 'magic')
        self.log_message("⚡ AI processing your data...", 'magic')
        self.status_label.config(text="🤖 AI Magic in progress...")
        
        # Implement magic button functionality
        self.perform_magic_button_analysis()

    @abstractmethod
    def perform_magic_button_analysis(self):
        pass

    def export_results(self, format_type):
        """Export with format-specific handling"""
        # Placeholder - implement actual export
        self.log_message(f"📄 Exporting as {format_type.upper()}...", 'info')
