#!/usr/bin/env python3
"""
SkhiBridgX Desktop Analytics - Core GUI Framework
Professional Tkinter application framework for sector analytics apps
"""

import tkinter as tk
from tkinter import ttk, filedialog, messagebox, scrolledtext
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import seaborn as sns
from datetime import datetime, timedelta
import os
import json
import threading
from abc import ABC, abstractmethod

class SectorApp(ABC):
    """Abstract base class for sector-specific analytics applications"""
    
    def __init__(self, root):
        self.root = root
        self.df = None
        self.reports_generated = []
        self.sector_name = self.get_sector_name()
        self.sector_color = self.get_sector_color()
        
        # Configure main window
        self.root.title(f"🚀 SkhiBridgX {self.sector_name} Analytics Pro - FREE TEST")
        
        # Responsive window sizing based on screen size
        screen_width = self.root.winfo_screenwidth()
        screen_height = self.root.winfo_screenheight()
        
        # Use 90% of screen size, with reasonable limits
        window_width = min(int(screen_width * 0.9), 1400)
        window_height = min(int(screen_height * 0.9), 900)
        
        # Center the window
        x = (screen_width - window_width) // 2
        y = (screen_height - window_height) // 2
        
        self.root.geometry(f"{window_width}x{window_height}+{x}+{y}")
        self.root.minsize(1000, 600)  # Minimum usable size
        self.root.configure(bg='#1e293b')
        
        self.setup_ui()
        self.populate_report_list()
        self.log_message(f"🎉 {self.sector_name} Analytics App Started - FREE TEST VERSION")
        self.log_message("💡 This app normally costs $199 - Free with TESTFREE coupon!")
        
    @abstractmethod
    def get_sector_name(self) -> str:
        """Return the sector name (e.g., 'Insurance', 'Real Estate', 'CPA')"""
        pass
        
    @abstractmethod
    def get_sector_color(self) -> str:
        """Return the sector's primary color (e.g., '#06b6d4')"""
        pass
        
    @abstractmethod
    def get_report_list(self) -> list:
        """Return list of 20+ reports for this sector"""
        pass
        
    @abstractmethod
    def load_sample_data(self):
        """Load sector-specific sample data"""
        pass
        
    @abstractmethod
    def perform_sector_analysis(self):
        """Perform sector-specific analytics with 20+ reports"""
        pass

    def setup_ui(self):
        """Create the main UI layout"""
        # Main title with sector branding
        title_frame = tk.Frame(self.root, bg='#1e293b')
        title_frame.pack(pady=20)
        
        # Sector-specific icon and title
        title = tk.Label(title_frame, 
                        text=f"{self.get_sector_icon()} SkhiBridgX {self.sector_name} Analytics Pro",
                        font=('Arial', 24, 'bold'),
                        fg=self.sector_color,
                        bg='#1e293b')
        title.pack()
        
        subtitle = tk.Label(title_frame,
                           text=f"Free Test Version - Process CSV Data • Generate 20+ {self.sector_name} Reports",
                           font=('Arial', 12),
                           fg='#94a3b8',
                           bg='#1e293b')
        subtitle.pack()
        
        # Main container with better proportions
        main_frame = tk.Frame(self.root, bg='#1e293b')
        main_frame.pack(expand=True, fill='both', padx=20, pady=10)
        
        # Left panel - File operations and reports (40% width)
        left_panel = tk.Frame(main_frame, bg='#334155', relief='raised', bd=2)
        left_panel.pack(side='left', fill='both', padx=(0, 10), pady=10)
        left_panel.configure(width=500)  # Fixed width
        
        # Right panel - Results (60% width)  
        right_panel = tk.Frame(main_frame, bg='#334155', relief='raised', bd=2)
        right_panel.pack(side='right', fill='both', expand=True, pady=10)
        
        self.setup_left_panel(left_panel)
        self.setup_right_panel(right_panel)
        self.setup_status_bar()

    def setup_left_panel(self, parent):
        """Setup left panel with file operations and report list"""
        # File loading section
        file_frame = tk.LabelFrame(parent, text="📁 Load Data", 
                                  font=('Arial', 12, 'bold'),
                                  fg='white', bg='#334155')
        file_frame.pack(padx=10, pady=10, fill='x')
        
        load_btn = tk.Button(file_frame, 
                            text=f"📂 Load {self.sector_name} CSV/Excel File",
                            command=self.load_csv_file,
                            bg=self.sector_color, fg='white',
                            font=('Arial', 11, 'bold'),
                            relief='flat',
                            pady=8)
        load_btn.pack(padx=10, pady=10, fill='x')
        
        self.file_label = tk.Label(file_frame, text="No file loaded",
                                  font=('Arial', 9), fg='#94a3b8', bg='#334155')
        self.file_label.pack(padx=10, pady=5)
        
        # Sample data section
        sample_frame = tk.LabelFrame(parent, text="🎯 Test Data", 
                                   font=('Arial', 12, 'bold'),
                                   fg='white', bg='#334155')
        sample_frame.pack(padx=10, pady=10, fill='x')
        
        sample_btn = tk.Button(sample_frame,
                              text=f"📊 Use Sample {self.sector_name} Data",
                              command=self.load_sample_data,
                              bg='#8b5cf6', fg='white',
                              font=('Arial', 11, 'bold'),
                              relief='flat',
                              pady=8)
        sample_btn.pack(padx=10, pady=10, fill='x')
        
        # Magic Button section
        magic_frame = tk.LabelFrame(parent, text="⚡ Magic Button", 
                                  font=('Arial', 12, 'bold'),
                                  fg='white', bg='#334155')
        magic_frame.pack(padx=10, pady=10, fill='x')
        
        magic_btn = tk.Button(magic_frame,
                             text=f"🔴 {self.get_magic_button_text()}",
                             command=self.magic_button_action,
                             bg='#ef4444', fg='white',
                             font=('Arial', 11, 'bold'),
                             relief='flat',
                             pady=8)
        magic_btn.pack(padx=10, pady=10, fill='x')
        
        # Reports section
        reports_frame = tk.LabelFrame(parent, text="📈 Generate Reports", 
                                    font=('Arial', 12, 'bold'),
                                    fg='white', bg='#334155')
        reports_frame.pack(padx=10, pady=10, fill='both', expand=True)
        
        # Report list with scrollbar
        list_frame = tk.Frame(reports_frame, bg='#334155')
        list_frame.pack(padx=10, pady=5, fill='both', expand=True)
        
        scrollbar = tk.Scrollbar(list_frame)
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        
        self.report_listbox = tk.Listbox(list_frame, 
                                        font=('Arial', 9),
                                        bg='#475569', fg='white',
                                        selectbackground=self.sector_color,
                                        yscrollcommand=scrollbar.set,
                                        height=12)
        self.report_listbox.pack(side=tk.LEFT, fill='both', expand=True)
        scrollbar.config(command=self.report_listbox.yview)
        
        generate_btn = tk.Button(reports_frame,
                               text="🚀 Generate All Reports (20+)",
                               command=self.generate_all_reports_threaded,
                               bg='#ef4444', fg='white',
                               font=('Arial', 11, 'bold'),
                               relief='flat',
                               pady=8)
        generate_btn.pack(padx=10, pady=10, fill='x')

    def setup_right_panel(self, parent):
        """Setup right panel with results and logs"""
        # Results area
        results_frame = tk.LabelFrame(parent, text="📊 Analysis Results & Logs", 
                                    font=('Arial', 12, 'bold'),
                                    fg='white', bg='#334155')
        results_frame.pack(padx=10, pady=10, fill='both', expand=True)
        
        self.results_text = scrolledtext.ScrolledText(results_frame,
                                                     font=('Courier', 9),
                                                     bg='#1e293b', fg='#94a3b8',
                                                     wrap=tk.WORD,
                                                     height=30)
        self.results_text.pack(padx=10, pady=10, fill='both', expand=True)
        
        # Export controls
        export_frame = tk.Frame(results_frame, bg='#334155')
        export_frame.pack(padx=10, pady=5, fill='x')
        
        # Quick export button (no directory selection)
        tk.Button(export_frame, text="⚡ Quick Export", 
                 bg='#8b5cf6', fg='white', font=('Arial', 10),
                 command=self.quick_export).pack(side='left', padx=5)
        
        tk.Button(export_frame, text="📄 Export PDF", 
                 bg='#06b6d4', fg='white', font=('Arial', 10),
                 command=lambda: self.export_results('pdf')).pack(side='left', padx=5)
        
        tk.Button(export_frame, text="📊 Export Excel", 
                 bg='#10b981', fg='white', font=('Arial', 10),
                 command=lambda: self.export_results('xlsx')).pack(side='left', padx=5)
        
        tk.Button(export_frame, text="🌐 Export HTML", 
                 bg='#f59e0b', fg='white', font=('Arial', 10),
                 command=lambda: self.export_results('html')).pack(side='left', padx=5)

    def setup_status_bar(self):
        """Setup status bar at bottom"""
        status_frame = tk.Frame(self.root, bg='#0f172a', relief='sunken', bd=1)
        status_frame.pack(side='bottom', fill='x')
        
        self.status_label = tk.Label(status_frame, 
                                   text=f"Ready - Load {self.sector_name} data or use sample data to begin",
                                   font=('Arial', 10), fg=self.sector_color, bg='#0f172a')
        self.status_label.pack(side='left', padx=10, pady=5)
        
        # Version info
        version_label = tk.Label(status_frame, 
                               text="v1.0.0-beta | Python 3.12 + R 4.3 | Quantum Analytics",
                               font=('Arial', 8), fg='#64748b', bg='#0f172a')
        version_label.pack(side='right', padx=10, pady=5)

    def populate_report_list(self):
        """Populate the report list with sector-specific reports"""
        reports = self.get_report_list()
        for report in reports:
            self.report_listbox.insert(tk.END, report)

    def get_sector_icon(self) -> str:
        """Return sector-specific emoji icon"""
        icons = {
            'Insurance': '🛡️',
            'Real Estate': '🏠', 
            'CPA': '📊'
        }
        return icons.get(self.sector_name, '⚡')

    def get_magic_button_text(self) -> str:
        """Return sector-specific magic button text"""
        magic_texts = {
            'Insurance': 'One-Click State Filing Pack',
            'Real Estate': 'Instant CMA Generator', 
            'CPA': 'Schedule C Optimizer'
        }
        return magic_texts.get(self.sector_name, 'Magic Analytics')

    def load_csv_file(self):
        """Load CSV or Excel file with enhanced error handling"""
        filetypes = [
            ("Data files", "*.csv;*.xlsx;*.xls"),
            ("CSV files", "*.csv"),
            ("Excel files", "*.xlsx;*.xls"),
            ("All files", "*.*")
        ]
        
        file_path = filedialog.askopenfilename(
            title=f"Select {self.sector_name} Data File",
            filetypes=filetypes
        )
        
        if file_path:
            self.log_message(f"📁 Attempting to load file: {os.path.basename(file_path)}")
            try:
                # Detect file type and load accordingly
                file_ext = os.path.splitext(file_path)[1].lower()
                
                if file_ext == '.csv':
                    # Try different encodings for CSV files
                    encodings_to_try = ['utf-8', 'latin-1', 'cp1252', 'iso-8859-1']
                    df_loaded = False
                    
                    for encoding in encodings_to_try:
                        try:
                            self.df = pd.read_csv(file_path, encoding=encoding)
                            self.log_message(f"✅ CSV loaded successfully with {encoding} encoding")
                            df_loaded = True
                            break
                        except UnicodeDecodeError:
                            continue
                        except Exception as e:
                            self.log_message(f"⚠️ Encoding {encoding} failed: {str(e)[:100]}...")
                            continue
                    
                    if not df_loaded:
                        raise Exception("Could not load CSV with any common encoding")
                        
                elif file_ext in ['.xlsx', '.xls']:
                    self.df = pd.read_excel(file_path, engine='openpyxl' if file_ext == '.xlsx' else 'xlrd')
                    self.log_message(f"✅ Excel file loaded successfully")
                else:
                    # Try CSV first, then Excel for unknown extensions
                    try:
                        self.df = pd.read_csv(file_path, encoding='utf-8')
                        self.log_message("✅ File loaded as CSV")
                    except:
                        self.df = pd.read_excel(file_path)
                        self.log_message("✅ File loaded as Excel")
                
                # Validate the loaded data
                if self.df.empty:
                    raise Exception("File appears to be empty")
                    
                # Clean column names (remove extra spaces, etc.)
                self.df.columns = self.df.columns.str.strip()
                
                # Log file statistics
                file_size = os.path.getsize(file_path) / 1024  # KB
                self.file_label.config(text=f"Loaded: {os.path.basename(file_path)}")
                self.status_label.config(text=f"✅ Loaded {len(self.df)} records from file")
                self.log_message(f"📊 Data shape: {self.df.shape[0]} rows, {self.df.shape[1]} columns")
                self.log_message(f"📁 File size: {file_size:.1f} KB")
                self.log_message(f"🔍 Columns: {', '.join(self.df.columns.tolist()[:10])}{'...' if len(self.df.columns) > 10 else ''}")
                
                # Check data quality
                missing_percent = (self.df.isnull().sum().sum() / (len(self.df) * len(self.df.columns))) * 100
                if missing_percent > 0:
                    self.log_message(f"⚠️ Data contains {missing_percent:.1f}% missing values")
                else:
                    self.log_message("✅ Data quality: No missing values detected")
                
                # Show a preview of the data
                self.log_message("\n📋 DATA PREVIEW:")
                preview_lines = self.df.head(3).to_string(max_cols=8, max_colwidth=15).split('\n')
                for line in preview_lines:
                    self.log_message(f"   {line}")
                if len(self.df) > 3:
                    self.log_message(f"   ... and {len(self.df) - 3} more rows")
                
            except Exception as e:
                error_msg = f"Failed to load file:\n\nFile: {os.path.basename(file_path)}\nError: {str(e)}"
                messagebox.showerror("File Load Error", error_msg)
                self.log_message(f"❌ Error loading file: {str(e)}")
                self.file_label.config(text="File load failed")
                self.status_label.config(text="❌ File load failed")

    def generate_all_reports_threaded(self):
        """Generate all reports in a background thread"""
        if self.df is None:
            messagebox.showwarning("No Data", "Please load data first or use sample data")
            return
        
        # Disable the button to prevent multiple clicks
        self.root.after(0, self.disable_generate_button)
        
        # Start background thread
        thread = threading.Thread(target=self.generate_all_reports, daemon=True)
        thread.start()

    def disable_generate_button(self):
        """Disable generate button during processing"""
        # Find and disable the generate button
        for child in self.root.winfo_children():
            self.find_and_disable_button(child, "Generate All Reports")

    def find_and_disable_button(self, widget, text):
        """Recursively find and disable button"""
        try:
            if isinstance(widget, tk.Button) and text in widget.cget('text'):
                widget.config(state='disabled', text="🔄 Generating Reports...")
                return
        except:
            pass
        
        for child in widget.winfo_children():
            self.find_and_disable_button(child, text)

    def enable_generate_button(self):
        """Re-enable generate button after processing"""
        for child in self.root.winfo_children():
            self.find_and_enable_button(child, "Generating Reports")

    def find_and_enable_button(self, widget, text):
        """Recursively find and enable button"""
        try:
            if isinstance(widget, tk.Button) and text in widget.cget('text'):
                widget.config(state='normal', text="🚀 Generate All Reports (20+)")
                return
        except:
            pass
        
        for child in widget.winfo_children():
            self.find_and_enable_button(child, text)

    def generate_all_reports(self):
        """Generate all sector-specific reports"""
        self.root.after(0, lambda: self.log_message("🚀 Starting comprehensive analytics..."))
        self.root.after(0, lambda: self.status_label.config(text="🔄 Generating 20+ professional reports..."))
        
        # Clear previous results
        self.root.after(0, lambda: self.results_text.delete(1.0, tk.END))
        self.reports_generated = []
        
        try:
            # Perform sector-specific analysis
            self.perform_sector_analysis()
            
            self.root.after(0, lambda: self.status_label.config(text="✅ All reports generated successfully!"))
            self.root.after(0, lambda: messagebox.showinfo("Success", 
                              f"Generated {len(self.reports_generated)} professional reports!\\n"
                              "Check the results panel for detailed analysis."))
                              
        except Exception as e:
            self.root.after(0, lambda: messagebox.showerror("Error", f"Analysis failed: {str(e)}"))
            self.root.after(0, lambda: self.status_label.config(text="❌ Analysis failed"))
            self.root.after(0, lambda: self.log_message(f"❌ Error: {str(e)}"))
        
        finally:
            self.root.after(0, self.enable_generate_button)

    def magic_button_action(self):
        """Execute sector-specific magic button action"""
        if self.df is None:
            messagebox.showwarning("No Data", "Please load data first or use sample data")
            return
            
        self.log_message(f"🔴 MAGIC BUTTON ACTIVATED: {self.get_magic_button_text()}")
        self.log_message("⚡ Performing automated sector-specific optimization...")
        
        # This will be implemented by each sector
        self.perform_magic_button_analysis()

    @abstractmethod
    def perform_magic_button_analysis(self):
        """Perform magic button analysis - implemented by each sector"""
        pass

    def quick_export(self):
        """Quick export to current directory (no dialog)"""
        if not self.reports_generated:
            messagebox.showwarning("No Results", "Please generate reports first")
            return
            
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        export_path = os.path.join(os.getcwd(), f"{self.sector_name}_Analytics_{timestamp}")
        
        try:
            os.makedirs(export_path, exist_ok=True)
            
            # Export all formats
            self.export_html(export_path)
            self.export_pdf(export_path)
            self.export_excel(export_path)
            
            self.log_message(f"⚡ Quick export complete: {export_path}")
            self.log_message("📄 Exported: report.html, report.txt, data_and_results.xlsx")
            messagebox.showinfo("Quick Export Complete", 
                f"All reports exported to:\n{export_path}\n\n"
                "Includes: HTML, PDF (text), and Excel formats")
                
        except Exception as e:
            # Try /tmp as fallback
            try:
                fallback_path = os.path.join("/tmp", f"{self.sector_name}_Analytics_{timestamp}")
                os.makedirs(fallback_path, exist_ok=True)
                
                self.export_html(fallback_path)
                self.export_pdf(fallback_path)
                self.export_excel(fallback_path)
                
                self.log_message(f"⚠️ Permission denied for current directory")
                self.log_message(f"⚡ Quick export to fallback: {fallback_path}")
                messagebox.showinfo("Quick Export Complete", 
                    f"Permission denied for current directory.\n\n"
                    f"Reports exported to:\n{fallback_path}\n\n"
                    "You can copy files from there to your preferred location.")
            except Exception as e2:
                self.log_message(f"❌ Quick export failed: {str(e2)}")
                messagebox.showerror("Quick Export Failed", f"Could not export results:\n{str(e2)}")
    
    def export_results(self, format_type):
        """Export results in specified format"""
        if not self.reports_generated:
            messagebox.showwarning("No Results", "Please generate reports first")
            return
        
        # Try to use a default directory that's accessible
        default_dirs = [
            os.path.expanduser("~/Documents"),
            os.path.expanduser("~/Downloads"),
            os.path.expanduser("~/Desktop"),
            os.path.expanduser("~"),
            os.getcwd(),
            "/tmp"
        ]
        
        initial_dir = None
        for dir_path in default_dirs:
            if os.path.exists(dir_path) and os.access(dir_path, os.W_OK):
                initial_dir = dir_path
                break
        
        try:
            output_dir = filedialog.askdirectory(
                title="Select Export Directory",
                initialdir=initial_dir
            )
        except Exception as e:
            self.log_message(f"⚠️ Dialog error: {str(e)}")
            # Fallback to current directory
            output_dir = os.getcwd()
            self.log_message(f"📁 Using current directory: {output_dir}")
        
        if not output_dir:
            # User cancelled, try current directory as fallback
            output_dir = os.getcwd()
            response = messagebox.askyesno(
                "Export Location", 
                f"No directory selected. Export to current directory?\n\n{output_dir}"
            )
            if not response:
                return
        
        try:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            export_path = os.path.join(output_dir, f"{self.sector_name}_Analytics_{timestamp}")
            os.makedirs(export_path, exist_ok=True)
            
            # Export based on format
            if format_type == 'html':
                self.export_html(export_path)
            elif format_type == 'pdf':
                self.export_pdf(export_path)  
            elif format_type == 'xlsx':
                self.export_excel(export_path)
                
            self.log_message(f"📄 Results exported to: {export_path}")
            messagebox.showinfo("Export Complete", f"Results exported to:\n{export_path}")
            
        except PermissionError:
            # Try alternative location
            fallback_path = os.path.join("/tmp", f"{self.sector_name}_Analytics_{timestamp}")
            try:
                os.makedirs(fallback_path, exist_ok=True)
                
                if format_type == 'html':
                    self.export_html(fallback_path)
                elif format_type == 'pdf':
                    self.export_pdf(fallback_path)  
                elif format_type == 'xlsx':
                    self.export_excel(fallback_path)
                
                self.log_message(f"⚠️ Permission denied for selected directory")
                self.log_message(f"📄 Results exported to fallback location: {fallback_path}")
                messagebox.showinfo("Export Complete", 
                    f"Permission denied for selected directory.\n\n"
                    f"Results exported to:\n{fallback_path}\n\n"
                    f"You can copy files from there to your preferred location.")
            except Exception as e:
                self.log_message(f"❌ Export failed: {str(e)}")
                messagebox.showerror("Export Failed", f"Could not export results:\n{str(e)}")
                
        except Exception as e:
            self.log_message(f"❌ Export failed: {str(e)}")
            messagebox.showerror("Export Failed", f"Could not export results:\n{str(e)}")

    def export_html(self, output_dir):
        """Export results as HTML"""
        html_content = f"""
        <!DOCTYPE html>
        <html>
        <head>
            <title>{self.sector_name} Analytics Results</title>
            <style>
                body {{ font-family: Arial, sans-serif; margin: 40px; background: #f5f5f5; }}
                .header {{ background: #1e293b; color: white; padding: 20px; border-radius: 8px; }}
                .results {{ background: white; padding: 20px; margin-top: 20px; border-radius: 8px; }}
                .watermark {{ color: #ef4444; font-weight: bold; }}
            </style>
        </head>
        <body>
            <div class="header">
                <h1>{self.get_sector_icon()} {self.sector_name} Analytics Results</h1>
                <p>Generated: {datetime.now().strftime("%Y-%m-%d %H:%M:%S")}</p>
                <p class="watermark">🔒 DEMO VERSION - WATERMARKED</p>
            </div>
            <div class="results">
                <pre>{self.results_text.get(1.0, tk.END)}</pre>
            </div>
        </body>
        </html>
        """
        
        with open(os.path.join(output_dir, "report.html"), "w", encoding="utf-8") as f:
            f.write(html_content)

    def export_pdf(self, output_dir):
        """Export results as PDF (placeholder - requires additional libraries)"""
        # For now, create a text file - PDF export would require reportlab or weasyprint
        with open(os.path.join(output_dir, "report.txt"), "w", encoding="utf-8") as f:
            f.write(f"{self.sector_name} Analytics Results\\n")
            f.write("=" * 50 + "\\n")
            f.write(f"Generated: {datetime.now()}\\n\\n")
            f.write("🔒 DEMO VERSION - WATERMARKED\\n\\n")
            f.write(self.results_text.get(1.0, tk.END))

    def export_excel(self, output_dir):
        """Export data and results as Excel"""
        if self.df is not None:
            with pd.ExcelWriter(os.path.join(output_dir, "data_and_results.xlsx")) as writer:
                # Export original data
                self.df.to_excel(writer, sheet_name="Original Data", index=False)
                
                # Export summary statistics
                if not self.df.select_dtypes(include=[np.number]).empty:
                    self.df.describe().to_excel(writer, sheet_name="Summary Stats")
                
                # Export results text as a simple table
                results_df = pd.DataFrame({
                    "Analysis Results": [self.results_text.get(1.0, tk.END)]
                })
                results_df.to_excel(writer, sheet_name="Results", index=False)

    def log_message(self, message):
        """Add message to results text area (thread-safe)"""
        def update_ui():
            timestamp = datetime.now().strftime("%H:%M:%S")
            formatted_message = f"[{timestamp}] {message}"
            self.results_text.insert(tk.END, formatted_message + "\\n")
            self.results_text.see(tk.END)
            
        # Schedule UI update on main thread
        self.root.after(0, update_ui)
